## packages needed
library(numDeriv)
library(ks)
library(LaplacesDemon)
library(lqmm)
library(Matrix)
library(pbapply)
library(parallel)
library(emulator)
library(tictoc)
library(MASS)
library(MBESS)
library(quantreg)
library(gtools)
library(maxLik)

library(ggplot2)
library(reshape2)
library(ggpubr)

### Function to estimate the maximum likelihood estimator
mle.MAL = function(y,X,ID,tau,parm.ini=NULL,epsi=1e-2){
  # y: vector with the outcomes
  # X: design matrix
  # ID: identification for the clusters (individuals)
  # parm.ini: initial values for parameters (Beta, diag. Delta, off-diagonal values of Psi)
  # epsi: epsilon parameter (epsi=0 two-stages estimator, epsi>0 direct maximization)
  
  ### functions needed
  # Functions for the reparameterization of the correlation matrix
  Theta2Corr = function(d,Thetaval){
    # p: dimension the correlation matrix
    # Thetaval: vector of the Theta matix (Vec Theta)
    ThetaMat = matrix(0,d,d)
    ThetaMat[lower.tri(ThetaMat)] =Thetaval
    cos.Theta = cos(ThetaMat)
    sin.Theta = sin(ThetaMat)
    U = matrix(0,d,d)
    for(j in 2:d){
      U[j,1:j]=cumprod(c(1,sin.Theta[j,1:(j-1)]))*cos.Theta[j,1:j]
    }
    U[1,1] = 1
    R = tcrossprod(U)
    return(R)
  }
  Corr2Theta = function(R){
    # p: dimension the correlation matrix
    # Thetaval: vector of the Theta matix (Vec Theta)
    B = chol(R)
    B = t(B)
    d = ncol(R)
    U = matrix(0,d,d)
    
    U[,1] = acos(B[,1])
    
    if(d==2){U[2,1] = acos(B[2,1])}else{
      for(i in 3:(d)){
        for(j in 2:(i-1)){
          #U[i,j]=acos(B[i,j]*(1/prod(sin(acos(B[i,1:(j-1)])))))
          U[i,j]=acos(B[i,j]*(1/prod(sin(U[i,1:(j-1)]))))
        }
      }
      
    }
    Theta = U[lower.tri(U)]
    return(Theta)
  }
  TargetFun = function(Thetaval,R,W){
    p=dim(R)[1]
    W.vec = ks::vech(W)
    R.adj = Theta2Corr(p,Thetaval)
    R.adj.vec = ks::vech(R.adj)
    R.vec = ks::vech(R)
    S = sum(W.vec*(R.adj.vec-R.vec)^2)
    return(S)
  }
  GenCor.Scaling <- function(R){
    p = ncol(R)
    is.pd = is.positive.definite(R)
    if(!is.pd){
      W = matrix(1,p,p)
      Sol = optim(par=rep(0,p*(p-1)/2),fn=TargetFun,R=R,W=W)
      Ralt = Theta2Corr(p,Sol$par)
    }else{Ralt=R}
    return(Ralt)
  }
  
  # FUnctions to compute the log-likelihood
  loglike.MAL = function(parm,Y,X,tau,epsi=0){
    p = ncol(X)
    if(is.matrix(Y)){
      d = nrow(Y)    
      N = ncol(Y)
    }else{
      d = length(Y)
      N=1
    }
    
    v = (2 - d)/2
    tau.vec =rep(tau,d)
    e =  (1-2*tau.vec)/(tau.vec*(1-tau.vec)) # constant
    delta =  sqrt(2/(tau.vec*(1-tau.vec))) # constant
    Beta = parm[1:p]
    ds = parm[(p+1):(d+p)]
    
    if(any(ds < 0)){return(NA)}
    rho = parm[p+d+1]
    if(rho < -1 | rho > 1){return(NA)}
    Psi = invvech(c(1,rho,1))
    Psi.inv = solve(Psi)
    
    mu = e*ds
    Mu = matrix(mu,d,N)
    Loc = matrix(c(X%*%Beta),d,N)
    Sigma.tilde = t(Psi*delta)*delta
    Sigma = as.symmetric.matrix(t(Sigma.tilde*ds)*ds)
    if(!is.positive.definite(Sigma)){
      Sigma = as.symmetric.matrix(make.positive.definite(Sigma))
    }
    Omega = as.inverse(Sigma)
    Omega.tilde = as.inverse(Sigma.tilde)
    
    Yc = Y - Loc
    
    
    m_i = colSums(Yc * crossprod(Omega,Yc))
    
    dd = 2+sum(mu * crossprod(Omega,mu))
    if(epsi==0){
      Const = log(2) - log(2 * pi) * (d/2)
    }else{
      Const = 0.5*log(2) - log(sqrt(epsi)) - log(besselK(sqrt(2*epsi),1)) - log(2 * pi) * (d/2)
    }
    T1 = -N*logdet(Sigma) * 0.5
    T2 =  sum(Yc * crossprod(Omega,Mu))
    if(v==0){
      T3=0
    }else{
      T3 = (v/2)*sum(log(m_i + epsi))    
    }
    T4 = -N*v*0.5*log(dd)
    T6 = sqrt(dd*(m_i+epsi))
    B.T6 = besselK(T6,v)
    T5 =sum(log(B.T6))
    N*Const+T1 + T2 + T3 + T4 + T5
  }
  loglike.MAL.2 = function(parm,Y,X,tau,epsi=0){
    p = ncol(X)
    
    if(is.matrix(Y)){
      d = nrow(Y)    
      N = ncol(Y)
    }else{
      d = length(Y)
      N=1
    }
    
    v = (2 - d)/2
    tau.vec =rep(tau,d)
    e =  (1-2*tau.vec)/(tau.vec*(1-tau.vec)) # constant
    delta =  sqrt(2/(tau.vec*(1-tau.vec))) # constant
    Beta = parm[1:p]
    ds = parm[(p+1):(d+p)]
    if(any(ds < 0)){return(NA)}
    thetas = parm[-c(1:(p+d))]
    if(any(thetas < 0) | any(thetas > pi)){return(NA)}
    Ds = diag(ds)
    Rho = Theta2Corr(d,thetas)
    loc = c(X%*%Beta)
    Yc = Y - matrix(loc,d,N,byrow = F)
    
    
    mu = ds*e
    Mu = matrix(mu,d,N,byrow = F)
    Sigma.tilde = t(Rho*delta)*delta
    Sigma = as.symmetric.matrix(t(Sigma.tilde*ds)*ds)
    if(!is.positive.definite(Sigma)){
      Sigma = as.symmetric.matrix(make.positive.definite(Sigma))
    }
    Omega = as.inverse(Sigma)
    x.Omega.x = colSums(Yc * crossprod(Omega,Yc))
    mu.Omega.mu = colSums(Mu * crossprod(Omega,Mu))
    x.Omega.mu = colSums(Yc * crossprod(Omega,Mu))
    
    
    if(epsi==0){
      Const = - log(2 * pi) * (d/2) + log(2)
    }else{
      Const = 0.5*log(2) - log(sqrt(epsi)) - log(besselK(sqrt(2*epsi),1)) - log(2 * pi) * (d/2)
    }    
    sum(Const + x.Omega.mu  - logdet(Sigma) * 0.5 + (log(x.Omega.x+epsi) - 
                                                       (log(2 + mu.Omega.mu))) *(v/2) + 
          log(besselK(sqrt((2 + mu.Omega.mu) * (x.Omega.x+epsi)),v)))
  }
  
  loglike.MAL.Beta = function(parm,Y,X,tau,epsi=0,ds,rho){
    p = ncol(X)
    if(is.matrix(Y)){
      d = nrow(Y)    
      N = ncol(Y)
    }else{
      d = length(Y)
      N=1
    }
    
    v = (2 - d)/2
    tau.vec =rep(tau,d)
    e =  (1-2*tau.vec)/(tau.vec*(1-tau.vec)) # constant
    delta =  sqrt(2/(tau.vec*(1-tau.vec))) # constant
    Beta = parm
    Psi = invvech(c(1,rho,1))
    Psi.inv = solve(Psi)
    
    mu = e*ds
    Mu = matrix(mu,d,N)
    Loc = matrix(c(X%*%Beta),d,N)
    Sigma.tilde = t(Psi*delta)*delta
    Sigma = as.symmetric.matrix(t(Sigma.tilde*ds)*ds)
    if(!is.positive.definite(Sigma)){
      Sigma = as.symmetric.matrix(make.positive.definite(Sigma))
    }
    Omega = as.inverse(Sigma)
    Omega.tilde = as.inverse(Sigma.tilde)
    
    Yc = Y - Loc
    
    
    m_i = colSums(Yc * crossprod(Omega,Yc))
    
    dd = 2+sum(mu * crossprod(Omega,mu))
    if(epsi==0){
      Const = log(2) - log(2 * pi) * (d/2)
    }else{
      Const = 0.5*log(2) - log(sqrt(epsi)) - log(besselK(sqrt(2*epsi),1)) - log(2 * pi) * (d/2)
    }
    T1 = -N*logdet(Sigma) * 0.5
    T2 =  sum(Yc * crossprod(Omega,Mu))
    if(v==0){
      T3=0
    }else{
      T3 = (v/2)*sum(log(m_i + epsi))    
    }
    T4 = -N*v*0.5*log(dd)
    T6 = sqrt(dd*(m_i+epsi))
    B.T6 = besselK(T6,v)
    T5 =sum(log(B.T6))
    N*Const+T1 + T2 + T3 + T4 + T5
  }
  loglike.MAL.OtherParm = function(parm,Y,X,tau,epsi=0,Beta){
    p = ncol(X)
    if(is.matrix(Y)){
      d = nrow(Y)    
      N = ncol(Y)
    }else{
      d = length(Y)
      N=1
    }
    
    v = (2 - d)/2
    tau.vec =rep(tau,d)
    e =  (1-2*tau.vec)/(tau.vec*(1-tau.vec)) # constant
    delta =  sqrt(2/(tau.vec*(1-tau.vec))) # constant
    ds = parm[1:d]
    
    if(any(ds < 0)){return(NA)}
    rho = parm[d+1]
    if(rho < -1 | rho > 1){return(NA)}
    Psi = invvech(c(1,rho,1))
    Psi.inv = solve(Psi)
    
    mu = e*ds
    Mu = matrix(mu,d,N)
    Loc = matrix(c(X%*%Beta),d,N)
    Sigma.tilde = t(Psi*delta)*delta
    Sigma = as.symmetric.matrix(t(Sigma.tilde*ds)*ds)
    if(!is.positive.definite(Sigma)){
      Sigma = as.symmetric.matrix(make.positive.definite(Sigma))
    }
    Omega = as.inverse(Sigma)
    Omega.tilde = as.inverse(Sigma.tilde)
    
    Yc = Y - Loc
    
    
    m_i = colSums(Yc * crossprod(Omega,Yc))
    
    dd = 2+sum(mu * crossprod(Omega,mu))
    if(epsi==0){
      Const = log(2) - log(2 * pi) * (d/2)
    }else{
      Const = 0.5*log(2) - log(sqrt(epsi)) - log(besselK(sqrt(2*epsi),1)) - log(2 * pi) * (d/2)
    }
    T1 = -N*logdet(Sigma) * 0.5
    T2 =  sum(Yc * crossprod(Omega,Mu))
    if(v==0){
      T3=0
    }else{
      T3 = (v/2)*sum(log(m_i + epsi))    
    }
    T4 = -N*v*0.5*log(dd)
    T6 = sqrt(dd*(m_i+epsi))
    B.T6 = besselK(T6,v)
    T5 =sum(log(B.T6))
    N*Const+T1 + T2 + T3 + T4 + T5
  }
  
  loglike.MAL.2.Beta = function(parm,Y,X,tau,epsi=0,ds,Rho){
    p = ncol(X)
    
    if(is.matrix(Y)){
      d = nrow(Y)    
      N = ncol(Y)
    }else{
      d = length(Y)
      N=1
    }
    
    v = (2 - d)/2
    tau.vec =rep(tau,d)
    e =  (1-2*tau.vec)/(tau.vec*(1-tau.vec)) # constant
    delta =  sqrt(2/(tau.vec*(1-tau.vec))) # constant
    Beta = parm
    Ds = diag(ds)
    loc = c(X%*%Beta)
    Yc = Y - matrix(loc,d,N,byrow = F)
    
    
    mu = ds*e
    Mu = matrix(mu,d,N,byrow = F)
    Sigma.tilde = t(Rho*delta)*delta
    Sigma = as.symmetric.matrix(t(Sigma.tilde*ds)*ds)
    if(!is.positive.definite(Sigma)){
      Sigma = as.symmetric.matrix(make.positive.definite(Sigma))
    }
    Omega = as.inverse(Sigma)
    x.Omega.x = colSums(Yc * crossprod(Omega,Yc))
    mu.Omega.mu = colSums(Mu * crossprod(Omega,Mu))
    x.Omega.mu = colSums(Yc * crossprod(Omega,Mu))
    
    
    if(epsi==0){
      Const = - log(2 * pi) * (d/2) + log(2)
    }else{
      Const = 0.5*log(2) - log(sqrt(epsi)) - log(besselK(sqrt(2*epsi),1)) - log(2 * pi) * (d/2)
    }    
    sum(Const + x.Omega.mu  - logdet(Sigma) * 0.5 + (log(x.Omega.x+epsi) - 
                                                       (log(2 + mu.Omega.mu))) *(v/2) + 
          log(besselK(sqrt((2 + mu.Omega.mu) * (x.Omega.x+epsi)),v)))
  }
  loglike.MAL.2.OtherParm = function(parm,Y,X,tau,epsi=0,Beta){
    p = ncol(X)
    
    if(is.matrix(Y)){
      d = nrow(Y)    
      N = ncol(Y)
    }else{
      d = length(Y)
      N=1
    }
    
    v = (2 - d)/2
    tau.vec =rep(tau,d)
    e =  (1-2*tau.vec)/(tau.vec*(1-tau.vec)) # constant
    delta =  sqrt(2/(tau.vec*(1-tau.vec))) # constant
    ds = parm[1:d]
    if(any(ds < 0)){return(NA)}
    thetas = parm[-c(1:d)]
    if(any(thetas < 0) | any(thetas > pi)){return(NA)}
    Ds = diag(ds)
    Rho = Theta2Corr(d,thetas)
    loc = c(X%*%Beta)
    Yc = Y - matrix(loc,d,N,byrow = F)
    
    
    mu = ds*e
    Mu = matrix(mu,d,N,byrow = F)
    Sigma.tilde = t(Rho*delta)*delta
    Sigma = as.symmetric.matrix(t(Sigma.tilde*ds)*ds)
    if(!is.positive.definite(Sigma)){
      Sigma = as.symmetric.matrix(make.positive.definite(Sigma))
    }
    Omega = as.inverse(Sigma)
    x.Omega.x = colSums(Yc * crossprod(Omega,Yc))
    mu.Omega.mu = colSums(Mu * crossprod(Omega,Mu))
    x.Omega.mu = colSums(Yc * crossprod(Omega,Mu))
    
    
    if(epsi==0){
      Const = - log(2 * pi) * (d/2) + log(2)
    }else{
      Const = 0.5*log(2) - log(sqrt(epsi)) - log(besselK(sqrt(2*epsi),1)) - log(2 * pi) * (d/2)
    }    
    sum(Const + x.Omega.mu  - logdet(Sigma) * 0.5 + (log(x.Omega.x+epsi) - 
                                                       (log(2 + mu.Omega.mu))) *(v/2) + 
          log(besselK(sqrt((2 + mu.Omega.mu) * (x.Omega.x+epsi)),v)))
  }

  # Functions to compute the gradient and the Hessian matrix
  Grad.central = function(parm,Y,X,tau,epsi=0){
    k=1e-4
    if(is.matrix(Y)){
      d = nrow(Y)    
    }else{
      d = length(Y)
    }
    
    Grad = mapply(function(x){
      parm.eval.1 = parm.eval.2 = parm
      parm.eval.1[x] = parm[x] + k
      parm.eval.2[x] = parm[x] - k
      if(d>2){
        ll.1 = loglike.MAL.2(parm.eval.1,Y,X,tau,epsi=epsi)
        ll.2 = loglike.MAL.2(parm.eval.2,Y,X,tau,epsi=epsi)
      }else{
        ll.1 = loglike.MAL(parm.eval.1,Y,X,tau,epsi=epsi)
        ll.2 = loglike.MAL(parm.eval.2,Y,X,tau,epsi=epsi)
      }
      deriv = (ll.1-ll.2)/(2*k)
      return(deriv)
    },x=1:length(parm))
    return(Grad)
  }
  Hessian.fun = function(parm,Y,X,tau,epsi=0){
    k=1e-4
    evals =   combinations(length(parm),2,repeats.allowed = T)
    d = nrow(Y)
    if(d > 2){
      ll = loglike.MAL.2(parm,Y,X,tau,epsi=epsi)    
    }else{
      ll = loglike.MAL(parm,Y,X,tau,epsi=epsi)
    }
    
    
    Hessian.vals = mapply(function(x){
      eval = evals[x,]
      if(length(unique(eval))==1){
        parm.pos = parm.neg = parm
        parm.pos[unique(eval)] = parm[unique(eval)] + k
        parm.neg[unique(eval)] = parm[unique(eval)] - k
        if(d > 2){
          ll.pos = loglike.MAL.2(parm.pos,Y,X,tau,epsi=epsi)
          ll.neg = loglike.MAL.2(parm.neg,Y,X,tau,epsi=epsi)
        }else{
          ll.pos = loglike.MAL(parm.pos,Y,X,tau,epsi=epsi)
          ll.neg = loglike.MAL(parm.neg,Y,X,tau,epsi=epsi)
        }
        fit = (ll.pos  + ll.neg - 2*ll)/k^2
      }else{
        parm.pos.pos = parm.pos.neg = parm.neg.neg = parm.neg.pos = parm
        parm.pos.pos[eval] = parm[eval] + k
        parm.pos.neg[eval] = parm[eval] + c(k,-k)
        parm.neg.neg[eval] = parm[eval] -k
        parm.neg.pos[eval] = parm[eval] + c(-k,k)
        if(d > 2){
          ll.pos.pos = loglike.MAL.2(parm.pos.pos,Y,X,tau,epsi=epsi)
          ll.pos.neg = loglike.MAL.2(parm.pos.neg,Y,X,tau,epsi=epsi)
          ll.neg.neg = loglike.MAL.2(parm.neg.neg,Y,X,tau,epsi=epsi)
          ll.neg.pos = loglike.MAL.2(parm.neg.pos,Y,X,tau,epsi=epsi)
        }else{
          ll.pos.pos = loglike.MAL(parm.pos.pos,Y,X,tau,epsi=epsi)
          ll.pos.neg = loglike.MAL(parm.pos.neg,Y,X,tau,epsi=epsi)
          ll.neg.neg = loglike.MAL(parm.neg.neg,Y,X,tau,epsi=epsi)
          ll.neg.pos = loglike.MAL(parm.neg.pos,Y,X,tau,epsi=epsi)
          
        }
        fit = (ll.pos.pos - ll.pos.neg - ll.neg.pos + ll.neg.neg)/(4*k^2)
      }
      return(fit)
    },x=1:nrow(evals))
    p = length(parm)
    Hessian = invvech(Hessian.vals)
    return(Hessian)
  }
  
  Grad.central.OtherParm = function(parm,Y,X,tau,epsi=0,Beta){
    k=1e-4
    if(is.matrix(Y)){
      d = nrow(Y)    
    }else{
      d = length(Y)
    }
    
    Grad = mapply(function(x){
      parm.eval.1 = parm.eval.2 = parm
      parm.eval.1[x] = parm[x] + k
      parm.eval.2[x] = parm[x] - k
      if(d>2){
        ll.1 = loglike.MAL.2.OtherParm(parm.eval.1,Y,X,tau,epsi=epsi,Beta)
        ll.2 = loglike.MAL.2.OtherParm(parm.eval.2,Y,X,tau,epsi=epsi,Beta)
      }else{
        ll.1 = loglike.MAL.OtherParm(parm.eval.1,Y,X,tau,epsi=epsi,Beta)
        ll.2 = loglike.MAL.OtherParm(parm.eval.2,Y,X,tau,epsi=epsi,Beta)
      }
      deriv = (ll.1-ll.2)/(2*k)
      return(deriv)
    },x=1:length(parm))
    return(Grad)
  }
  
  if(!is.vector(y)){return('y is not a vector')}
  if(!is.matrix(X)){return('X is not a matrix')}
  if(tau > 1 | tau < 0){return('tau is not between 0 and 1')}
  
  d= unique(table(ID))
  N = length(y)/d
  p = ncol(X)
  Y = matrix(y,d,N,byrow = F)
  
  # computing initial values
  if(is.null(parm.ini)){
    v= (2 - d)/2
    tau.vec = rep(tau,d)
    e =  (1-2*tau.vec)/(tau.vec*(1-tau.vec)) # constant
    delta =  sqrt(2/(tau.vec*(1-tau.vec))) # constant
    
    Var.Y = var(t(Y))    
    ds.ini = sqrt(diag(Var.Y)/( e^2 + delta^2 ))
    y.temp = y - rep(ds.ini*e,N)
    Beta.ini = lm(y.temp~X-1)$coef
    DD = 1/(ds.ini*delta)
    Inner = Var.Y - t(tcrossprod(e)*ds.ini)*ds.ini
    rho.ini = as.symmetric.matrix(t(Inner*DD)*DD)
    if(!is.positive.definite(rho.ini)){
      rho.ini = GenCor.Scaling(rho.ini)
    }
    if(d==2){
      rho.ini = rho.ini[upper.tri(rho.ini)]
      parm.ini = c(Beta.ini,ds.ini,rho.ini)    
      if(is.infinite(loglike.MAL(parm.ini,Y,X,tau,epsi=epsi)) | 
         is.na(loglike.MAL(parm.ini,Y,X,tau,epsi=epsi)) ){
        parm.ini = c(Beta.ini,ds.ini,rep(0,length(rho.ini)))
      }
      mm.est = parm.ini
    }else{
      theta.ini = Corr2Theta(rho.ini)
      parm.ini = c(Beta.ini,ds.ini,theta.ini)    
      if(is.infinite(sum(loglike.MAL.2(parm.ini,Y,X,tau,epsi=epsi))) | 
         anyNA(loglike.MAL.2(parm.ini,Y,X,tau,epsi=epsi)) ){
        parm.ini = c(Beta.ini,ds.ini,rep(pi/2,length(theta.ini)))
      }
      mm.est = c(Beta.ini,ds.ini,c(rho.ini[upper.tri(rho.ini)]))
    }
    
  }else{mm.est = parm.ini}
  
  if(epsi>0){
    # direct maximization of the log-likelihood
    if(d==2){
      Fit = tryCatch(optim(parm.ini,loglike.MAL,gr=Grad.central,method='BFGS',control=list(maxit=5000,fnscale=-1),Y=Y,X=X,
                           tau=tau,epsi=epsi),error=function(e){e})
    }else{
      Fit = tryCatch(optim(parm.ini,loglike.MAL.2,gr=Grad.central,method='BFGS',control=list(maxit=5000,fnscale=-1),Y=Y,X=X,
                           tau=tau,epsi=epsi),error=function(e){e})
      if(Fit$convergence > 2){
        return(Fit)
      }
    }
    
    if(is.null(Fit)){
      Fit$estimates = rep(NA,7)
      Fit$code = 1
      Fit$iterations = NA
      Fit$maximum = NA
    }
    
    Est = Fit$par
    
    Beta.H = Est[1:p]
    names(Beta.H) = paste('B',0:(p-1),sep = '')
    ds.H = Est[c(p+1):(p+d)]
    names(ds.H) = paste('d',1:d,sep = '')
    if(d==2){
      Psi.H = diag(d)
      Psi.H[upper.tri(Psi.H)] = Est[-c(1:(p+d))]
      Psi.H = as.matrix(forceSymmetric(Psi.H))
    }else{
      Psi.H = Theta2Corr(d,Est[-c(1:(p+d))])
    }
    
    A.inv = Hessian.fun(Fit$par,Y=Y,X=X,tau=tau,epsi=epsi)
    grad.final = Grad.central(Fit$par,Y=Y,X=X,tau=tau,epsi=epsi)
    PD.A = tryCatch(is.positive.definite(-A.inv),error=function(e){FALSE})
    # Computing the covariance matrix of the estimates
    if(PD.A){
      A = solve(A.inv)
      B = mapply(function(x){
        X.i = X[ID==x,]
        y.i = Y[,x]
        b.i = Grad.central(Est,y.i,X.i,tau,epsi=epsi)
        B.i = tcrossprod(b.i,b.i)
        return(vech(B.i))
      },x=1:N)
      B = invvech(apply(B,1,sum))
      V = A%*%B%*%A
    }else{
      V = matrix(NA,nrow(A.inv),ncol(A.inv))
    }
    
    Control = c(Fit$convergence,Fit$counts[1],Fit$value)
    names(Control) = c('Convergence','Iterations','logLikelihood')
    
    Estimates = list(Beta=Beta.H,ds = ds.H,Psi = Psi.H,V=V,Control=Control,
                     Deriv = list(Gradient = grad.final,Hessian = A.inv))
  }else{
    # maximization of the log-likelihood in two steps (epsi=0)
    step = 0
    tol=200
    lls = 0
    parmH=parm.ini
    parm.test = parm.ini
    if(d==2){
      while(tol > 1e-8 & step < 50){
        fit1 = optim(parmH[-c(1:p)],loglike.MAL.OtherParm,gr=Grad.central.OtherParm,method='BFGS',control=list(maxit=5000,fnscale=-1),Y=Y,X=X,
                     tau=tau,epsi=0,Beta=parmH[1:p])
        
        parm.test[-c(1:p)] = fit1$par
        rho = parm.test[p+d+1]
        fit2 = optim(parm.test[c(1:p)],loglike.MAL.Beta,gr=NULL,method='Nelder-Mead',control=list(maxit=5000,fnscale=-1),Y=Y,X=X,
                     tau=tau,epsi=0,ds=parm.test[(p+1):(p+d)],rho=rho)
        parm.test[c(1:p)] = fit2$par
        parm.diff = parm.test - parmH
        eval.diff = abs(fit2$value - lls)
        tol = min(crossprod(parm.diff),eval.diff)
        
        parmH = parm.test
        lls=fit2$value
        step = step +1
      }
      
    }else{
      while(tol > 1e-8 & step < 50){
        fit1 = optim(parmH[-c(1:p)],loglike.MAL.2.OtherParm,gr=Grad.central.OtherParm,method='BFGS',control=list(maxit=5000,fnscale=-1),Y=Y,X=X,
                     tau=tau,epsi=0,Beta=parmH[1:p])
        
        parm.test[-c(1:p)] = fit1$par
        Rho = Theta2Corr(d,parm.test[(p+d+1):(length(parm.test))])
        fit2 = optim(parm.test[c(1:p)],loglike.MAL.2.Beta,gr=NULL,method='Nelder-Mead',control=list(maxit=5000,fnscale=-1),Y=Y,X=X,
                     tau=tau,epsi=0,ds=parm.test[(p+1):(p+d)],Rho=Rho)
        parm.test[c(1:p)] = fit2$par
        parm.diff = parm.test - parmH
        eval.diff = abs(fit2$value - lls)
        tol = max(crossprod(parm.diff),eval.diff)
        
        parmH = parm.test
        lls=fit2$value
        step = step +1
      }
    }
    
    Beta.H = parmH[1:p]
    names(Beta.H) = paste('B',0:(p-1),sep = '')
    ds.H = parmH[c(p+1):(p+d)]
    names(ds.H) = paste('d',1:d,sep = '')
    if(d==2){
      Psi.H = diag(d)
      Psi.H[upper.tri(Psi.H)] = parmH[-c(1:(p+d))]
      Psi.H = as.matrix(forceSymmetric(Psi.H))
    }else{
      Psi.H = Theta2Corr(d,parmH[-c(1:(p+d))])
    }
    V = matrix(NA,length(parmH),length(parmH))
    Control = c(fit2$convergence,step,fit2$value)
    Gradient = rep(NA,length(parmH))
    Hessian = matrix(NA,length(parmH),length(parmH))
    Estimates = list(Beta=Beta.H,ds = ds.H,Psi = Psi.H,V=V,Control=Control,
                     Deriv = list(Gradient = Gradient,Hessian = Hessian))
  }
  # the output consists of a list with [1] estimates of Beta, [2] estimates of the diagonal elements of the Delta matrix, 
  # [3] estimate of the correlation matrix, [4] covariance matrix of the estimates, [5] Control: convergence, iterations and log-likelihood, 
  # [6] Gradient and Hessian
  return(Estimates)
}

### Function to simulate the data
# Normal, Student-t, Cauchy
genData = function(N,BetaMat,Sigma,dist='NORM',Seed=NULL){
  # N: Number of clusters
  # BetaMat: Matrix (or vector) of location parameters
  # Sigma: Variance-covariance matrix
  # dist: multivariate distribution (NORM: normal, CAUCHY: Cauchy, MT: Student-t)
  # Seed: seed for simulations
  if(!is.null(Seed)){
    set.seed(Seed)
  }
  if(is.matrix(BetaMat)){
    q = nrow(BetaMat)
  }else{
    q = 1
  }
  t = ncol(Sigma)/q
  ClusterID = rep(1:N,each=q*t)  
  if(dist=='NORM'){
    er = mvrnorm(N,rep(0,t*q),Sigma)  
  }else{
    if(dist=='CAUCHY'){
      er = rmvc(N,rep(0,t*q),Sigma)
    }else{
      if(dist=='MT'){
        er = rmvt(N, rep(0,t*q), Sigma*(1/3), df=3)
      }else{
        return(c('distribution not available'))
      }
    }
  }
  Beta = vec(BetaMat,byrow=T)
  Z = rbinom(1,N,0.5)
  g  = c(rep(0,(N-Z)),rep(1,Z))
  x = seq(0,1,length.out = t)
  X = mapply(function(i){
    X = cbind(1,x,x*g[i])
    kronecker(diag(q),X)
  },i=1:N,SIMPLIFY = F)
  Xq = do.call(rbind,X)
  Loc = Xq%*%Beta
  
  Y = Loc + vec(er,byrow=T)
  
  
  Data = cbind(ClusterID,Y,Xq)
  p = ncol(Xq)
  colnames(Data) = c('ID','Y',paste('X',0:(p-1),sep = ''))
  return(as.data.frame(Data))
}
# multivariate asymmetric Laplace
genData.MAL = function(N,Beta,d.values,Psi,tau.val,Seed=NULL){
  # N: Number of clusters
  # Beta: vector of location parameters
  # d.values = diagonal values of the Delta matrix
  # Psi: matrix Psi.
  # tau.val: value for tau
  # Seed: seed for simulations
  if(!is.null(Seed)){
    set.seed(Seed)
  }
  d = length(d.values)  
  e =  rep((1-2*tau.val)/(tau.val*(1-tau.val)),d) # constant
  delta =  rep(sqrt(2/(tau.val*(1-tau.val))),d) # constant
  
  Mu = d.values*e
  Sigma.tilde = t(Psi*delta)*delta
  Sigma = as.symmetric.matrix(t(Sigma.tilde*d.values)*d.values)
  
  er = raml(N,Mu,Sigma)
  
  x = (0:(d-1))/(d-1)
  X1 = cbind(1,x,x)
  X2 = cbind(1,x,0)
  N1 = rbinom(1,N,0.5)
  N2 = N - N1
  Loc1 = X1%*%Beta
  Loc2 = X2%*%Beta
  Y1 = er[1:N1,] +  matrix(Loc1,N1,d,byrow=T)
  Y2 = er[-c(1:N1),] +  matrix(Loc2,N2,d,byrow=T)
  
  X = rbind(matrix(ks::vec(X1,byrow = T),N1*d,ncol(X1),byrow = T),
            matrix(ks::vec(X2,byrow = T),N2*d,ncol(X2),byrow = T))
  Y = rbind(Y1,Y2)
  ClusterID = rep(1:N,each=d)
  Data = cbind(ClusterID,c(t(Y)),X)
  p = ncol(X)
  colnames(Data) = c('ID','Y',paste('X',0:(p-1),sep = ''))
  return(as.data.frame(Data))
}

############ Functions to simulate data and estimate parameters (check inside the functions the folders to save the results)
### Function to estimate the MLE to simulated data
mle.Sim = function(M,N,tau,rho,epsi,dist='NORM',q=1,save.file=T){
  # M: number of simulations
  # N: number of clusters per simulation
  # tau: tau parameter (MAL distribution) or quantile required (other distributions)
  # epsi: parameter epsilon
  # dist: distribution (NORM: normal, MT: Student-t, CAUCHY: Cauchy, MAL: multivariate asymmetric Laplace)
  # q: 1 or 2 (number of bi-variate variables), q=2 is only available for NORM and MT
  # save.file: save a .csv file with the simulations
  
  if(dist=='MAL'){
    Beta = c(4,2,1)
    d.values = c(0.25,0.5)
    Psi = invvech(c(1,rho,1))      
    
    
    cl <- makeCluster(getOption("cl.cores", 2))
    clusterExport(cl=cl, varlist=c('M','N','q','tau','rho','dist','Beta','Psi','d.values','epsi',
                                   'genData.MAL','genData','mle.MAL'), envir=environment())
    
    clusterEvalQ(cl=cl,library('tictoc'))
    clusterEvalQ(cl=cl,library('numDeriv'))
    clusterEvalQ(cl=cl,library('ks'))
    clusterEvalQ(cl=cl,library('LaplacesDemon'))
    clusterEvalQ(cl=cl,library('lqmm'))
    clusterEvalQ(cl=cl,library('Matrix'))
    clusterEvalQ(cl=cl,library('MASS'))
    clusterEvalQ(cl=cl,library('MBESS'))
    clusterEvalQ(cl=cl,library('quantreg'))
    clusterEvalQ(cl=cl,library('maxLik'))
    clusterEvalQ(cl=cl,library('gtools'))
    Fit = pblapply(1:M,function(x){
      Seed=122+x
      Data = genData.MAL(N,Beta,d.values,Psi,tau,Seed)
      y = Data$Y
      X = as.matrix(Data[,-c(1:2)])
      ID = Data$ID
      tic()
      mle = tryCatch(mle.MAL.3(y,X,ID,tau,parm.ini=NULL,epsi=epsi),
                     error=function(e){NULL})
      exec.time = toc(quiet =T)
      if(is.null(mle)){
        if(q==1){
          output = rep(NA,17)
          return(output)
        }else{
          output = rep(NA,46)
          return(output)
        }
        
      }      
      time = exec.time$toc - exec.time$tic
      EstH = c(mle$Beta,mle$ds,mle$Psi[lower.tri(mle$Psi)])
      V = mle$V
      PD.V = tryCatch(is.positive.definite(V),error=function(e){FALSE})      
      if(PD.V){
        CI.inf = EstH[1:3] + qnorm(0.025)*sqrt(diag(V))[1:3]
        CI.sup = EstH[1:3] + qnorm(0.975)*sqrt(diag(V))[1:3]        
      }else{
        CI.inf = rep(NA,3)
        CI.sup = rep(NA,3)  
      }
      CI = cbind(CI.inf,CI.sup)
      
      
      output = c(EstH,diag(V),c(t(CI)),PD.V,time)
      return(c(output))
    },cl=cl)
    stopCluster(cl)
    MLE = do.call('rbind',Fit)
    
    
    if(q==1){
      LABEL = c(paste('B',0:2,sep = ''),'d1','d2','psi')
      LABEL = c(LABEL,paste('V',LABEL,sep='.'),
                paste('CI',rep(LABEL[1:3],each=2),sep='.'), 'PD.V','time')
      Case = paste('N',N,'tau',tau,'rho',rho,'epsi',epsi,'txt',sep = '.')
    }else{
      
      LABEL = c(paste('B',0:5,sep = ''),paste('d',1:4,sep=''),'psi12','psi13','psi14',
                'psi23','psi24','psi34')
      LABEL = c(LABEL,paste('V',LABEL,sep='.'), paste('CI',rep(LABEL[1:6],each=2),sep='.'),
                'PD.V','time') 
      Case = paste('q',2,'N',N,'tau',tau,'rho',rho,'epsi',epsi,'txt',sep = '.')
    }
    
    colnames(MLE) = LABEL    
    
    if(save.file){
      write.csv(MLE,paste('Estimates.MAL/mle.',Case,sep = ''),row.names = F)
    }
    
    return(MLE)
    
  }else{
    
    if(q==1){
      Beta = c(4,2,1)    
      Sigma = invvech(c(1,2*rho,4))    
    }else{
      Beta = matrix(c(4,2,1,5,-3,1),nrow=2,byrow = T)
      Vars = c(1,4,1,4)
      Corr.t = invvech(c(1,rho,1))
      Corr.q = invvech(c(1,rho,1))
      Sigma = cor2cov(kronecker(Corr.t,Corr.q),sqrt(Vars))  
    }
    
    
    cl <- makeCluster(getOption("cl.cores", 3))
    clusterExport(cl=cl, varlist=c('M','N','q','tau','rho','dist','Beta','Sigma','epsi',
                                   'genData','mle.MAL'), envir=environment())
    clusterEvalQ(cl=cl,library('tictoc'))
    clusterEvalQ(cl=cl,library('numDeriv'))
    clusterEvalQ(cl=cl,library('ks'))
    clusterEvalQ(cl=cl,library('LaplacesDemon'))
    clusterEvalQ(cl=cl,library('lqmm'))
    clusterEvalQ(cl=cl,library('Matrix'))
    clusterEvalQ(cl=cl,library('MASS'))
    clusterEvalQ(cl=cl,library('MBESS'))
    clusterEvalQ(cl=cl,library('quantreg'))
    clusterEvalQ(cl=cl,library('maxLik'))
    clusterEvalQ(cl=cl,library('gtools'))
    Fit = pblapply(1:M,function(x){
      Seed=122+x
      Data = genData(N,Beta,Sigma,dist,Seed)
      p = length(Beta)
      y = Data$Y
      X = as.matrix(Data[,-c(1:2)])
      ID = Data$ID
      tic()
      mle = tryCatch(mle.MAL.3(y,X,ID,tau,parm.ini=NULL,epsi=epsi),
                     error=function(e){NULL})
      exec.time = toc(quiet =T)
      if(is.null(mle)){
        if(q==1){
          output = rep(NA,17)
          return(output)
        }else{
          output = rep(NA,46)
          return(output)
        }
        
      }
      time = exec.time$toc - exec.time$tic
      EstH = c(mle$Beta,mle$ds,mle$Psi[lower.tri(mle$Psi)])
      V = mle$V
      PD.V = tryCatch(is.positive.definite(V),error=function(e){FALSE})      
      if(PD.V){
        CI.inf = EstH[1:p] + qnorm(0.025)*sqrt(diag(V))[1:p]
        CI.sup = EstH[1:p] + qnorm(0.975)*sqrt(diag(V))[1:p]        
      }else{
        CI.inf = rep(NA,p)
        CI.sup = rep(NA,p)  
      }
      CI = cbind(CI.inf,CI.sup)
      
      output = c(EstH,diag(V),c(t(CI)),PD.V,time)        
      return(c(output))
    },cl=cl)
    stopCluster(cl)
    MLE = do.call('rbind',Fit)
    if(q==1){
      LABEL = c(paste('B',0:2,sep = ''),'d1','d2','psi')
      LABEL = c(LABEL,paste('V',LABEL,sep='.'),
                paste('CI',rep(LABEL[1:3],each=2),sep='.'),
                'PD.V','time')
      Case = paste('N',N,'tau',tau,'rho',rho,'epsi',epsi,'txt',sep = '.')
    }else{
      
      LABEL = c(paste('B',0:5,sep = ''),paste('d',1:4,sep=''),'psi12','psi13','psi14',
                'psi23','psi24','psi34')
      LABEL = c(LABEL,paste('V',LABEL,sep='.'), paste('CI',rep(LABEL[1:6],each=2),sep='.'),
                'PD.V','time') 
      Case = paste('q',2,'N',N,'tau',tau,'rho',rho,'epsi',epsi,'txt',sep = '.')
    }
    
    colnames(MLE) = LABEL
    
    if(save.file){
      write.csv(MLE,paste('Estimates.',dist,'/mle.',Case,sep = ''),row.names = F) 
    }
    return(MLE)
  }
}
## Function to estimate the UQR to simulated data
uqr.Sim = function(M,N,tau,rho,dist='NORM',q=1,save.file=T){
  # M: number of simulations
  # N: number of clusters per simulation
  # tau: quantile required
  # dist: distribution (NORM: normal, MT: t-student, CAUCHY: cauchy)
  # q: 1 or 2 (number of bi-variate variables), q=2 is only available for NORM and MT
  # save.file: save a .csv file with the simulations
  
  if(q==1){
    Beta = c(4,2,1)    
    Sigma = invvech(c(1,2*rho,4))    
  }else{
    Beta = matrix(c(4,2,1,5,-3,1),nrow=2,byrow = T)
    Vars = c(1,4,1,4)
    Corr.t = invvech(c(1,rho,1))
    Corr.q = invvech(c(1,rho,1))
    Sigma = cor2cov(kronecker(Corr.t,Corr.q),sqrt(Vars))  
  }
  cl <- makeCluster(getOption("cl.cores", 2))
  clusterExport(cl=cl, varlist=c('M','N','q','tau','rho','dist','Beta','Sigma',
                                 'genData.MAL','genData'), envir=environment())
  clusterEvalQ(cl=cl,library('tictoc'))
  clusterEvalQ(cl=cl,library('numDeriv'))
  clusterEvalQ(cl=cl,library('ks'))
  clusterEvalQ(cl=cl,library('LaplacesDemon'))
  clusterEvalQ(cl=cl,library('lqmm'))
  clusterEvalQ(cl=cl,library('Matrix'))
  clusterEvalQ(cl=cl,library('MASS'))
  clusterEvalQ(cl=cl,library('MBESS'))
  clusterEvalQ(cl=cl,library('quantreg'))
  clusterEvalQ(cl=cl,library('maxLik'))
  clusterEvalQ(cl=cl,library('gtools'))
  Fit = pblapply(1:M,function(x){
    Seed=122+x
    Data = genData(N,Beta,Sigma,dist,Seed)
    y = Data$Y
    X = as.matrix(Data[,-c(1:2)])
    ID = Data$ID
    tic()
    uqr = rq(y~X-1,tau)
    exec.time = toc(quiet =T)
    time = exec.time$toc - exec.time$tic
    fit = c(uqr$coefficients,time)
    return(c(fit))
  },cl=cl)
  stopCluster(cl)
  Fit = do.call('rbind',Fit)
  if(q==1){
    LABEL = paste('B',0:2,sep = '') 
    Case = paste('N',N,'tau',tau,'rho',rho,'txt',sep = '.')
    
  }else{
    LABEL = paste('B',0:5,sep = '')
    Case = paste('q',q,'N',N,'tau',tau,'rho',rho,'txt',sep = '.')
    
  }
  
  colnames(Fit) = c(LABEL,'time')
  
  if(save.file){
    write.csv(Fit,paste('Estimates.',dist,'/uqr.',Case,sep = ''),row.names = F)
  }
  return(Fit)
}
### Function to estimate the MME to simulated data
mm.Sim = function(M,N,tau,rho,dist='NORM',save.file=T){
  # M: number of simulations
  # N: number of clusters per simulation
  # tau: tau parameter (MAL distribution) or quantile required (other distributions)
  # dist: distribution (NORM: normal, MT: t-student, CAUCHY: cauchy, MAL: multivariate asymetric Laplace)
  # save.file: save a .csv file with the simulations
  
  Beta = c(4,2,1)    
  Sigma = invvech(c(1,2*rho,4))
  Case = paste('N',N,'tau',tau,'rho',rho,'epsi',epsi,'txt',sep = '.')
  
  if(dist=='MAL'){
    d.values = c(0.25,0.5)
    Psi = invvech(c(1,rho,1))
    
    cl <- makeCluster(getOption("cl.cores", 3))
    clusterExport(cl=cl, varlist=c('M','N','q','tau','rho','dist','grad','Beta','Psi','d.values','epsi',
                                   'loglike.MAL','genData.MAL','genData','mle.MAL','GenCor.Scaling',
                                   'TargetFun','Corr2Theta','Theta2Corr',
                                   'Grad.central','Hessian.fun'), envir=environment())
    clusterEvalQ(cl=cl,library('tictoc'))
    clusterEvalQ(cl=cl,library('numDeriv'))
    clusterEvalQ(cl=cl,library('ks'))
    clusterEvalQ(cl=cl,library('LaplacesDemon'))
    clusterEvalQ(cl=cl,library('lqmm'))
    clusterEvalQ(cl=cl,library('Matrix'))
    clusterEvalQ(cl=cl,library('MASS'))
    clusterEvalQ(cl=cl,library('MBESS'))
    clusterEvalQ(cl=cl,library('quantreg'))
    clusterEvalQ(cl=cl,library('maxLik'))
    clusterEvalQ(cl=cl,library('gtools'))
    
    Fit = pblapply(1:M,function(x){
      Seed=122+x
      Data = genData.MAL(N,Beta,d.values,Psi,tau,Seed)
      y = Data$Y
      X = as.matrix(Data[,-c(1:2)])
      ID = Data$ID
      d= unique(table(ID))
      N = length(y)/d
      p = ncol(X)
      Y = matrix(y,d,N,byrow = F)
      
      v= (2 - d)/2
      tau.vec = rep(tau,d)
      e =  (1-2*tau.vec)/(tau.vec*(1-tau.vec)) # constant
      delta =  sqrt(2/(tau.vec*(1-tau.vec))) # constant
      
      Var.Y = var(t(Y))    
      ds.ini = sqrt(diag(Var.Y)/( e^2 + delta^2 ))
      y.temp = y - rep(ds.ini*e,N)
      Beta.ini = lm(y.temp~X-1)$coef
      DD = 1/(ds.ini*delta)
      Inner = Var.Y - t(tcrossprod(e)*ds.ini)*ds.ini
      rho.ini = as.symmetric.matrix(t(Inner*DD)*DD)
      if(!is.positive.definite(rho.ini)){
        rho.ini = GenCor.Scaling(rho.ini)
      }
      output = c(Beta.ini,ds.ini,rho.ini[upper.tri(rho.ini)])
      return(c(output))
    },cl=cl)
    stopCluster(cl)
    Fit = do.call('rbind',Fit)
    LABEL = c(paste('B',0:2,sep = ''),'d1','d2','psi')
    colnames(Fit) = LABEL
    
    if(save.file){
      write.csv(Fit,paste('Estimates.',dist,'/uqr.',Case,sep = ''),row.names = F)
    }
    return(Fit)
    
  }else{
    
    cl <- makeCluster(getOption("cl.cores", 2))
    clusterExport(cl=cl, varlist=c('M','N','q','tau','rho','dist','grad','Beta','Sigma','epsi',
                                   'loglike.MAL','genData.MAL','genData','mle.MAL','GenCor.Scaling',
                                   'TargetFun','Corr2Theta','Theta2Corr',
                                   'Grad.central','Hessian.fun'), envir=environment())
    clusterEvalQ(cl=cl,library('tictoc'))
    clusterEvalQ(cl=cl,library('numDeriv'))
    clusterEvalQ(cl=cl,library('ks'))
    clusterEvalQ(cl=cl,library('LaplacesDemon'))
    clusterEvalQ(cl=cl,library('lqmm'))
    clusterEvalQ(cl=cl,library('Matrix'))
    clusterEvalQ(cl=cl,library('MASS'))
    clusterEvalQ(cl=cl,library('MBESS'))
    clusterEvalQ(cl=cl,library('quantreg'))
    clusterEvalQ(cl=cl,library('maxLik'))
    clusterEvalQ(cl=cl,library('gtools'))
    Fit = pblapply(1:M,function(x){
      Seed=122+x
      Data = genData(N,Beta,Sigma,dist,Seed)
      y = Data$Y
      X = as.matrix(Data[,-c(1:2)])
      ID = Data$ID
      
      d= unique(table(ID))
      N = length(y)/d
      p = ncol(X)
      Y = matrix(y,d,N,byrow = F)
      
      v= (2 - d)/2
      tau.vec = rep(tau,d)
      e =  (1-2*tau.vec)/(tau.vec*(1-tau.vec)) # constant
      delta =  sqrt(2/(tau.vec*(1-tau.vec))) # constant
      
      Var.Y = var(t(Y))    
      ds.ini = sqrt(diag(Var.Y)/( e^2 + delta^2 ))
      y.temp = y - rep(ds.ini*e,N)
      Beta.ini = lm(y.temp~X-1)$coef
      DD = 1/(ds.ini*delta)
      Inner = Var.Y - t(tcrossprod(e)*ds.ini)*ds.ini
      rho.ini = as.symmetric.matrix(t(Inner*DD)*DD)
      if(!is.positive.definite(rho.ini)){
        rho.ini = GenCor.Scaling(rho.ini)
      }
      output = c(Beta.ini,ds.ini,rho.ini[upper.tri(rho.ini)])
      return(c(output))
    },cl=cl)
    stopCluster(cl)
    Fit = do.call('rbind',Fit)
    LABEL = c(paste('B',0:2,sep = ''),'d1','d2','psi')
    colnames(Fit) = LABEL
    
    if(save.file){
      write.csv(Fit,paste('Estimates.',dist,'/uqr.',Case,sep = ''),row.names = F)
    }
    return(Fit)
  }
}

### Function to compare MLE vs UQR (or MME) by tau, epsilon, distribution and q
Compare.estimators.paper = function(tau,epsi=.005,dist='NORM',q=1){
  # tau: tau parameter (MAL distribution) or quantile required (other distributions)
  # epsi: parameter epsilon 
  # dist: distribution (NORM: normal, MT: t-student, CAUCHY: cauchy)
  # q: 1 or 2 (number of bi-variate variables), q=2 is only available for NORM and MT
  if(q==1){
    Beta = c(4,2,1)
    p=3
    var.ind = 7:9
    ci.ind = 13
    
  }else{
    Beta = c(4,2,1,5,-3,1)
    p=6
    var.ind = 17:22
    ci.ind =33
  }
  #   eps = c(1e-4,1e-3,0.01,0.05,0.1,0.5)
  
  rho = rep(c(0.5,0.9),each=3)
  N = rep(c(50,100,200),2)
  
  if(dist=='MAL'){
    parm = mapply(function(x){c(Beta,0.25,0.5,rho[x])},x=1:length(rho),SIMPLIFY = F)
    nparm = length(parm[[1]])
  }else{
    if(dist=='CAUCHY'){parm = Beta + c(1,1,0)*qcauchy(tau)}
    if(dist=='NORM'){parm = Beta + c(1,1,0)*qnorm(tau)}
    if(dist=='MT'){parm = Beta + c(sqrt(1/3),sqrt(4/3) - sqrt(1/3),0)*qt(tau,3)}
    nparm = length(parm)
  }
  
  
  if(q==1){
    MLE = mapply(function(x){
      est = tryCatch(read.csv(paste('Estimates.',dist,'/mle.N.',N[x],'.tau.',tau,'.rho.',rho[x],'.epsi.',epsi,'.txt',sep='')),
                     error=function(e){matrix(NA,1000,33)})
      est
    },x=1:length(N),SIMPLIFY = F)
  }else{
    MLE = mapply(function(x){
      est = tryCatch(read.csv(paste('Estimates.',dist,'/mle.q.2.N.',N[x],'.tau.',tau,'.rho.',rho[x],'.epsi.',epsi,'.txt',sep='')),
                     error=function(e){matrix(NA,1000,33)})
      est
    },x=1:length(N),SIMPLIFY = F)
    
  }
  
  if(dist=='MAL'){
    UQR = mapply(function(x){
      est = tryCatch(read.csv(paste('Estimates.',dist,'/mm.N.',N[x],'.tau.',tau,'.rho.',rho[x],'.txt',sep='')),
                     error=function(e){matrix(NA,1000,33)})
      est
    },x=1:length(N),SIMPLIFY = F)    
  }else{
    if(q==1){
      UQR = mapply(function(x){
        est = tryCatch(read.csv(paste('Estimates.',dist,'/uqr.N.',N[x],'.tau.',tau,'.rho.',rho[x],'.txt',sep='')),
                       error=function(e){matrix(NA,1000,33)})
        est
      },x=1:length(N),SIMPLIFY = F)      
    }else{
      UQR = mapply(function(x){
        est = tryCatch(read.csv(paste('Estimates.',dist,'/uqr.q.2.N.',N[x],'.tau.',tau,'.rho.',rho[x],'.txt',sep='')),
                       error=function(e){matrix(NA,1000,33)})
        est
      },x=1:length(N),SIMPLIFY = F)
    }
    
  } 
  
  
  BIAS = mapply(function(x){
    if(dist=='MAL'){parm_x= parm[[x]]}else{parm_x = parm}
    est = MLE[[x]]
    100*(colMeans(est[,1:nparm],na.rm=T)-parm_x)/parm_x
  },x=1:length(MLE))
  
  colnames(BIAS) = paste('rho',rho,'N',N,sep='.')
  
  MSE = mapply(function(x){
    if(dist=='MAL'){parm_x= parm[[x]]}else{parm_x = parm}
    est = MLE[[x]]
    M = nrow(est)
    (colMeans((est[,1:nparm]-matrix(parm_x,M,nparm,byrow = T))^2,na.rm=T))
  },x=1:length(MLE))
  
  MSE.UQR = mapply(function(x){
    if(dist=='MAL'){parm_x= parm[[x]]}else{parm_x = parm}
    est = UQR[[x]]
    M = nrow(est)
    (colMeans((est[,1:nparm]-matrix(parm_x,M,nparm,byrow = T))^2,na.rm=T))
  },x=1:length(MLE))
  
  RE = MSE.UQR/MSE
  
  colnames(RE) = colnames(BIAS)
  
  VarRatio= mapply(function(x){
    if(q==1){ind.x=6}else{ind.x=16}
    Vemp = diag(var(MLE[[x]][,1:nparm]))
    Vest.1 = colMeans(MLE[[x]][,ind.x + 1:nparm],na.rm=T)
    R.1 = Vest.1/Vemp
    PD.1 = mean(!apply(MLE[[x]][,ind.x + 1:nparm],1,anyNA))
    c(R.1,PD.1)        
    
  },x=1:length(MLE))
  
  colnames(VarRatio) = colnames(BIAS)
  
  COVER= mapply(function(x){
    if(dist=='MAL'){parm_x= parm[[x]]}else{parm_x = parm}
    if(q==1){
      ind.x=6
      cover.B0 = mean(MLE[[x]][ind.x*2 + 1] < parm_x[1] & MLE[[x]][ind.x*2 + 2] > parm_x[1], na.rm=T )
      cover.B1 = mean(MLE[[x]][ind.x*2 + 3] < parm_x[2] & MLE[[x]][ind.x*2 + 4] > parm_x[2], na.rm=T )
      cover.B2 = mean(MLE[[x]][ind.x*2 + 5] < parm_x[3] & MLE[[x]][ind.x*2 + 6] > parm_x[3], na.rm=T )
      c(cover.B0,cover.B1,cover.B2)
    }else{
      ind.x=16
      cover.B0 = mean(MLE[[x]][ind.x*2 + 1] < parm_x[1] & MLE[[x]][ind.x*2 + 2] > parm_x[1], na.rm=T )
      cover.B1 = mean(MLE[[x]][ind.x*2 + 3] < parm_x[2] & MLE[[x]][ind.x*2 + 4] > parm_x[2], na.rm=T )
      cover.B2 = mean(MLE[[x]][ind.x*2 + 5] < parm_x[3] & MLE[[x]][ind.x*2 + 6] > parm_x[3], na.rm=T )
      cover.B3 = mean(MLE[[x]][ind.x*2 + 7] < parm_x[4] & MLE[[x]][ind.x*2 + 8] > parm_x[4], na.rm=T )
      cover.B4 = mean(MLE[[x]][ind.x*2 + 9] < parm_x[5] & MLE[[x]][ind.x*2 + 10] > parm_x[5], na.rm=T )
      cover.B5 = mean(MLE[[x]][ind.x*2 + 11] < parm_x[6] & MLE[[x]][ind.x*2 + 12] > parm_x[6], na.rm=T )
      c(cover.B0,cover.B1,cover.B2,cover.B3,cover.B4,cover.B5)
      
    }
    
    
  },x=1:length(MLE))
  
  colnames(COVER) = colnames(BIAS)
  
  return(list(BIAS=BIAS,RE=RE,VarRatio=VarRatio,COVER=COVER))
}


### setting simulation scenarios
N = rep(c(50,100,200),each=9)
tau = rep(rep(c(0.25,0.5,0.9),each=3),3)
rho = rep(c(0.25,0.5,0.9),9)
# Case 1: bivariate asymmetric Laplace
mapply(function(x){
  mle.Sim(1000,N[x],tau[x],rho[x],epsi=0,dist='MAL',q=1,save.file=T)  
  mle.Sim(1000,N[x],tau[x],rho[x],epsi=0.01,dist='MAL',q=1,save.file=T)
  mm.Sim(1000,N[x],tau[x],rho[x],dist='MAL', save.file=T)
},x=1:27)
# Case 2: bivariate normal
mapply(function(x){
  mle.Sim(1000,N[x],tau[x],rho[x],epsi=0,dist='NORM',q=1,save.file=T)  
  mle.Sim(1000,N[x],tau[x],rho[x],epsi=0.01,dist='NORM',q=1,save.file=T)  
  uqr.Sim(1000,N[x],tau[x],rho[x],dist='NORM',q=1,save.file=T)
},x=1:27)
# Case 3: bivariate Student-t
mapply(function(x){
  mle.Sim(1000,N[x],tau[x],rho[x],epsi=0,dist='MT',q=1,save.file=T)  
  mle.Sim(1000,N[x],tau[x],rho[x],epsi=0.01,dist='MT',q=1,save.file=T)  
  uqr.Sim(1000,N[x],tau[x],rho[x],dist='MT',q=1,save.file = T)
},x=1:27)
# Case 4: bivariate Cauchy
mapply(function(x){
  mle.Sim(1000,N[x],tau[x],rho[x],epsi=0,dist='CAUCHY',q=1,save.file=T)  
  mle.Sim(1000,N[x],tau[x],rho[x],epsi=0.01,dist='CAUCHY',q=1,save.file=T)  
  uqr.Sim(1000,N[x],tau[x],rho[x],dist='CAUCHY',q=1)
},x=1:27)
# Case 5: four-dimensional normal
mapply(function(x){
  mle.Sim(1000,N[x],tau[x],rho[x],epsi=0,dist='NORM',q=2,save.file=T)  
  mle.Sim(1000,N[x],tau[x],rho[x],epsi=0.01,'NORM',dist='NORM',q=2,save.file=T)  
  uqr.Sim(1000,N[x],tau[x],rho[x],dist='NORM',q=2)
},x=1:27)
# Case 6: four-dimensional Student-t
mapply(function(x){
  mle.Sim(1000,N[x],tau[x],rho[x],epsi=0,dist='MT',q=2,save.file = T)  
  mle.Sim(1000,N[x],tau[x],rho[x],epsi=0.01,dist='MT',q=2,save.file = T)  
  uqr.Sim(1000,N[x],tau[x],rho[x],dist='MT',q=2,save.file = T)
},x=1:27)


###### Table with results
tau.x = rep(c(0.25,0.5,0.9),each=2)
eps.x = rep(c(0,0.01),3)


# Normal case
NORM = mapply(function(x){
  Compare.estimators.paper(tau=tau.x[x],eps=eps.x[x],dist='NORM',q=1)
},x=1:6,SIMPLIFY = F) 

# Normal - Bias and RE
NORM.BIAS = mapply(function(x){
  ind = x*2-1
  cbind(NORM[[ind+1]]$BIAS[,1:3],NA,NORM[[ind+1]]$BIAS[,4:6])
},x=1:3,SIMPLIFY = F)

NORM.RE = mapply(function(x){
  ind = x*2-1
  cbind(NORM[[ind+1]]$RE[,1:3],NA,NORM[[ind+1]]$RE[,4:6])
},x=1:3,SIMPLIFY = F)

NORM.BIAS = do.call(rbind,NORM.BIAS)
NORM.RE = do.call(rbind,NORM.RE)
NORM.Results = cbind(NORM.BIAS,NA,NORM.RE)

# Normal - Coverage
NORM.COVER = mapply(function(x){
  ind = x*2-1
  cbind(NORM[[ind+1]]$COVER[,1:3],NA,NORM[[ind+1]]$COVER[,4:6])
},x=1:3,SIMPLIFY = F)
NORM.COVER = do.call(rbind,NORM.COVER)

### student-t case
MT = mapply(function(x){
  Compare.estimators.paper(tau=tau.x[x],eps=eps.x[x],dist='MT',q=1)
},x=1:6,SIMPLIFY = F) 

# student-t - Bias and RE
MT.BIAS = mapply(function(x){
  ind = x*2-1
  cbind(MT[[ind+1]]$BIAS[,1:3],NA,MT[[ind+1]]$BIAS[,4:6])
},x=1:3,SIMPLIFY = F)

MT.RE = mapply(function(x){
  ind = x*2-1
  cbind(MT[[ind+1]]$RE[,1:3],NA,MT[[ind+1]]$RE[,4:6])
},x=1:3,SIMPLIFY = F)

MT.BIAS = do.call(rbind,MT.BIAS)
MT.RE = do.call(rbind,MT.RE)
MT.Results = cbind(MT.BIAS,NA,MT.RE)

# student-t - Coverage
MT.COVER = mapply(function(x){
  ind = x*2-1
  cbind(MT[[ind+1]]$COVER[,1:3],NA,MT[[ind+1]]$COVER[,4:6])
},x=1:3,SIMPLIFY = F)
MT.COVER = do.call(rbind,MT.COVER)

## Table 1 bias and relative efficiency for the Normal and student-t cases
Table.1 = rbind(NORM.Results,MT.Results)
xtable(Table.1)
## Table 2 coverage for the Normal and student-t cases
Table.2 = rbind(NORM.COVER,MT.COVER)
xtable(Table.2)

# MAL case (Supplementary Materials)
MAL = mapply(function(x){
  Compare.estimators.paper(tau=tau.x[x],eps=eps.x[x],dist='MAL',q=1)
},x=1:6,SIMPLIFY = F) 

# Table B1 MAL case - Bias
MAL.BIAS = mapply(function(x){
  ind = x*2-1
  cbind(MAL[[ind]]$BIAS[,1:3],NA,MAL[[ind]]$BIAS[,4:6],NA,
        MAL[[ind+1]]$BIAS[,1:3],NA,MAL[[ind+1]]$BIAS[,4:6])
},x=1:3,SIMPLIFY = F)

MAL.BIAS = do.call(rbind,MAL.BIAS)
xtable(MAL.BIAS)
# Table B2 MAL case - RE
MAL.RE = mapply(function(x){
  ind = x*2-1
  cbind(MAL[[ind]]$RE[,1:3],NA,MAL[[ind]]$RE[,4:6],NA,
        MAL[[ind+1]]$RE[,1:3],NA,MAL[[ind+1]]$RE[,4:6])
},x=1:3,SIMPLIFY = F)

MAL.RE = do.call(rbind,MAL.RE)
xtable(MAL.RE)

### Table B3 MAL case - Coverage
MAL.COVER = mapply(function(x){
  ind = x*2-1
  cbind(MAL[[ind+1]]$COVER[,1:3],NA,MAL[[ind+1]]$COVER[,4:6])
},x=1:3,SIMPLIFY = F)
MAL.COVER = do.call(rbind,MAL.COVER)
xtable(MAL.COVER)


### Cauchy case
CAUCHY = mapply(function(x){
  Compare.estimators.paper(tau=tau.x[x],eps=eps.x[x],dist='CAUCHY',q=1)
},x=1:6,SIMPLIFY = F) 

CAUCHY.BIAS = mapply(function(x){
  ind = x*2-1
  cbind(CAUCHY[[ind+1]]$BIAS[,1:3],NA,CAUCHY[[ind+1]]$BIAS[,4:6])
},x=1:3,SIMPLIFY = F)

CAUCHY.RE = mapply(function(x){
  ind = x*2-1
  cbind(CAUCHY[[ind+1]]$RE[,1:3],NA,CAUCHY[[ind+1]]$RE[,4:6])
},x=1:3,SIMPLIFY = F)

## Table C4
CAUCHY.BIAS = do.call(rbind,CAUCHY.BIAS)
CAUCHY.RE = do.call(rbind,CAUCHY.RE)
CAUCHY.Results = rbind(CAUCHY.BIAS,NA,CAUCHY.RE)

xtable(CAUCHY.Results,digits=4)


### 4d normal case
NORM4 = mapply(function(x){
  Compare.estimators.paper(tau=tau.x[x],eps=eps.x[x],dist='NORM',q=2)
},x=1:6,SIMPLIFY = F) 

# 4d normal case - Bias and RE
NORM4.BIAS = mapply(function(x){
  ind = x*2-1
  cbind(NORM4[[ind+1]]$BIAS[,1:3],NA,NORM4[[ind+1]]$BIAS[,4:6])
},x=1:3,SIMPLIFY = F)

NORM4.BIAS = do.call(rbind,NORM4.BIAS)

NORM4.RE = mapply(function(x){
  ind = x*2-1
  cbind(NORM4[[ind+1]]$RE[,1:3],NA,NORM4[[ind+1]]$RE[,4:6])
},x=1:3,SIMPLIFY = F)

NORM4.RE = do.call(rbind,NORM4.RE)

NORM4.COVER = mapply(function(x){
  ind = x*2-1
  cbind(NORM4[[ind+1]]$COVER[,1:3],NA,NORM4[[ind+1]]$COVER[,4:6])*100
},x=1:3,SIMPLIFY = F)
NORM4.COVER = do.call(rbind,NORM4.COVER)


NORM4.Results = cbind(NORM4.BIAS,NA,NORM4.RE)

### 4d t-student case
MT4 = mapply(function(x){
  Compare.estimators.paper(tau=tau.x[x],eps=eps.x[x],dist='MT',q=2)
},x=1:6,SIMPLIFY = F) 

# Table B.5 4d student-t case - Bias and RE
MT4.BIAS = mapply(function(x){
  ind = x*2-1
  cbind(MT4[[ind+1]]$BIAS[,1:3],NA,MT4[[ind+1]]$BIAS[,4:6])
},x=1:3,SIMPLIFY = F)

MT4.BIAS = do.call(rbind,MT4.BIAS)

MT4.RE = mapply(function(x){
  ind = x*2-1
  cbind(MT4[[ind+1]]$RE[,1:3],NA,MT4[[ind+1]]$RE[,4:6])
},x=1:3,SIMPLIFY = F)
MT4.RE = do.call(rbind,MT4.RE)

MT4.COVER = mapply(function(x){
  ind = x*2-1
  cbind(MT4[[ind+1]]$COVER[,1:3],NA,MT4[[ind+1]]$COVER[,4:6])*100
},x=1:3,SIMPLIFY = F)
MT4.COVER = do.call(rbind,MT4.COVER)


MT4.Results = cbind(MT4.BIAS,NA,MT4.RE)


## Table D5 relative bias and efficiency for the four-dimensional cases
Table.D5 = rbind(NORM4.Results,MT4.Results)
xtable(Table.D5)

# Table B.6 4d normal and 4d student-t cases - Coverage
xtable(cbind(NORM4.COVER,MT4.COVER))

##### Relative efficiency (MLE with epsilon=0 vs epsilon=0.01)
NORM.RE.epsi = mapply(function(x){
  ind = x*2-1
  cbind(NORM[[ind+1]]$RE[,1:3]/NORM[[ind]]$RE[,1:3],NA,
        NORM[[ind+1]]$RE[,4:6]/NORM[[ind]]$RE[,4:6])
},x=1:3,SIMPLIFY = F)

NORM.RE.epsi = do.call(rbind,NORM.RE.epsi)

NORM4.RE.epsi = mapply(function(x){
  ind = x*2-1
  cbind(NORM4[[ind+1]]$RE[,1:3]/NORM4[[ind]]$RE[,1:3],NA,
        NORM4[[ind+1]]$RE[,4:6]/NORM4[[ind]]$RE[,4:6])
},x=1:3,SIMPLIFY = F)

NORM4.RE.epsi = do.call(rbind,NORM4.RE.epsi)

MT.RE.epsi = mapply(function(x){
  ind = x*2-1
  cbind(MT[[ind+1]]$RE[,1:3]/MT[[ind]]$RE[,1:3],NA,
        MT[[ind+1]]$RE[,4:6]/MT[[ind]]$RE[,4:6])
},x=1:3,SIMPLIFY = F)

MT.RE.epsi = do.call(rbind,MT.RE.epsi)

MT4.RE.epsi = mapply(function(x){
  ind = x*2-1
  cbind(MT4[[ind+1]]$RE[,1:3]/MT4[[ind]]$RE[,1:3],NA,
        MT4[[ind+1]]$RE[,4:6]/MT4[[ind]]$RE[,4:6])
},x=1:3,SIMPLIFY = F)

MT4.RE.epsi = do.call(rbind,MT4.RE.epsi)

# Table E.7 Normal and Student-t cases. Relative efficiency MLE with epsilon=0 vs epsilon=0.01
xtable(cbind(NORM.RE.epsi,NA,MT.RE))
xtable(cbind(NORM4.RE.epsi,NA,MT4.RE))
